<?php

namespace App\Http\Controllers\APIs;

use App\Http\Controllers\Controller;
use App\Http\Requests\AttendanceRequest;
use App\Models\Attendance;
use App\Models\Conference;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;

class AttendanceController extends Controller
{
    public function index(): JsonResponse
    {
        return response()->json(User::all());
    }

    // ✅ 1. Fetch all registered users from Conference
    public function getUsers(): JsonResponse
    {
        $users = Conference::select('id', 'full_name')->get();
        return response()->json($users, 200);
    }

    // ✅ 2. Store Attendance (User must exist in Conference)
    public function register(Request $request)
{
    try {
        $validated = $request->validate([
            'conference_id' => 'required|exists:conferences,id',
            'arrival_time' => 'required|date_format:Y-m-d H:i:s',
            // 'departure_time' => 'nullable|date_format:Y-m-d H:i:s',
            'comments' => 'nullable|string|max:255',
        ]);

        // ✅ Check if user already submitted attendance
        $existingAttendance = Attendance::where('conference_id', $validated['conference_id'])->first();

        if ($existingAttendance) {
            return response()->json([
                'message' => 'You have already submitted attendance.',
                'data' => $existingAttendance
            ], 200);
        }

        // ✅ Create a new attendance record if it doesn't exist
        $attendance = Attendance::create([
            'conference_id' => $validated['conference_id'],
            'arrival_time' => $validated['arrival_time'],
            // 'departure_time' => $validated['departure_time'],
            'comments' => $validated['comments'],
            'status' => 'pending', // ✅ Default status
        ]);

        return response()->json([
            'message' => 'Attendance recorded successfully',
            'data' => $attendance
        ], 201);

    } catch (\Exception $e) {
        return response()->json([
            'error' => 'Failed to record attendance',
            'message' => $e->getMessage()
        ], 500);
    }
}

    // ✅ 3. Fetch All Attendances
    public function fetchAll(): JsonResponse
    {
        $attendances = Attendance::with('conference:id,full_name')->get();
        return response()->json($attendances, 200);
    }

    // ✅ 3. Acknowledge Attendances Sheet
    public function admin($id)
    {
        try {
            $attendance = Attendance::find($id);

            if (!$attendance) {
                return response()->json(['error' => 'Attendance record not found'], 404);
            }

            if ($attendance->status === 'acknowledged') {
                return response()->json(['message' => 'Attendance is already acknowledged'], 200);
            }

            $attendance->update(['status' => 'acknowledged']);

            return response()->json([
                'message' => 'Attendance acknowledged successfully!',
                'data' => $attendance
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Server error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

}
